import 'package:flutter/material.dart';
import 'package:get/get.dart';

import '../../../../common/ui.dart';
import '../../../routes/app_routes.dart';
import '../../global_widgets/text_field_widget.dart';
import '../controllers/wallet_form_controller.dart';

class WalletFormView extends GetView<WalletFormController> {
  @override
  Widget build(BuildContext context) {
    controller.walletForm = new GlobalKey<FormState>();
    return Scaffold(
        appBar: AppBar(
          title: Obx(() {
            return Text(
              controller.isCreateForm() ? "Add New Wallet".tr : controller.wallet.value.name,
              style: context.textTheme.titleLarge,
            );
          }),
          centerTitle: true,
          backgroundColor: Colors.transparent,
          automaticallyImplyLeading: false,
          leading: new IconButton(
            icon: new Icon(Icons.arrow_back_ios, color: Get.theme.hintColor),
            onPressed: () => Get.offAndToNamed(Routes.WALLETS),
          ),
          actions: [
            if (!controller.isCreateForm())
              new IconButton(
                padding: EdgeInsets.symmetric(horizontal: 20),
                icon: new Icon(
                  Icons.delete_outline,
                  color: Colors.redAccent,
                  size: 28,
                ),
                onPressed: () {
                  if (controller.wallet.value.balance > 0) {
                    Get.showSnackbar(Ui.ErrorSnackBar(message: "You can't delete non empty wallet".tr));
                  } else {
                    _showDeleteDialog(context);
                  }
                },
              ),
          ],
          elevation: 0,
        ),
        bottomNavigationBar: Container(
          padding: EdgeInsets.symmetric(vertical: 10),
          decoration: BoxDecoration(
            color: Get.theme.primaryColor,
            borderRadius: BorderRadius.only(topLeft: Radius.circular(20), topRight: Radius.circular(20)),
            boxShadow: [
              BoxShadow(color: Get.theme.focusColor.withOpacity(0.1), blurRadius: 10, offset: Offset(0, -5)),
            ],
          ),
          child: Row(
            children: [
              Expanded(
                child: MaterialButton(
                  onPressed: () {
                    if (controller.isCreateForm()) {
                      controller.createWalletForm();
                    } else {
                      controller.updateWalletForm();
                    }
                  },
                  padding: EdgeInsets.symmetric(horizontal: 30, vertical: 12),
                  shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(10)),
                  color: Get.theme.colorScheme.secondary,
                  child: Text("Save".tr, style: Get.textTheme.bodyMedium!.merge(TextStyle(color: Get.theme.primaryColor))),
                  elevation: 0,
                  hoverElevation: 0,
                  focusElevation: 0,
                  highlightElevation: 0,
                ),
              ),
            ],
          ).paddingSymmetric(vertical: 10, horizontal: 20),
        ),
        body: Form(
          key: controller.walletForm,
          child: SingleChildScrollView(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text("Wallet".tr, style: Get.textTheme.headlineSmall).paddingOnly(top: 25, bottom: 0, right: 22, left: 22),
                Text("Fill the following details to add new wallet".tr, style: Get.textTheme.bodySmall).paddingSymmetric(horizontal: 22, vertical: 5),
                TextFieldWidget(
                  onSaved: (input) {
                    controller.wallet.update((val) {
                      val?.name = input;
                    });
                    controller.wallet.value.name = input;
                  },
                  validator: (input) => input!.length < 1 ? "Field is required".tr : null,
                  initialValue: controller.wallet.value.name,
                  hintText: "My Wallet".tr,
                  labelText: "Wallet Name".tr,
                ),
              ],
            ),
          ),
        ));
  }

  void _showDeleteDialog(BuildContext context) {
    showDialog<void>(
      context: context,
      barrierDismissible: false, // user must tap button!
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(
            "Delete Wallet".tr,
            style: TextStyle(color: Colors.redAccent),
          ),
          content: SingleChildScrollView(
            child: Column(
              children: <Widget>[
                Text("This wallet will removed from your account".tr, style: Get.textTheme.bodyLarge),
              ],
            ),
          ),
          actions: <Widget>[
            TextButton(
              child: Text("Cancel".tr, style: Get.textTheme.bodyLarge),
              onPressed: () {
                Get.back();
              },
            ),
            TextButton(
              child: Text(
                "Confirm".tr,
                style: TextStyle(color: Colors.redAccent),
              ),
              onPressed: () {
                Get.back();
                controller.deleteWallet(controller.wallet.value);
              },
            ),
          ],
        );
      },
    );
  }
}
